#!/usr/bin/perl
# 
# Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      rootcrs.pl
#
#    DESCRIPTION
#      Clusterware configuration script for Oracle Clusterware home
#
#    NOTES
#      This is run once during configuration or patching of Oracle
#      clusterware home.
#      This script does the following:
#      1) Setup permissions of binaries and DSOs in Oracle Clusterware home
#      2) Setup OLR for storing Oracle local registry data
#      3) Setup GPnP wallet and profile
#      4) Setup and copy files for CLSINIT daemon
#      5) Start CLSINIT daemon
#      6) Copy required CRS resources for CLSINIT to start
#
#      NOTE: This is run during Oracle Clusterware home configurations only.
#            It should not be used for any other homes such as SI HA home,
#            or Oracle RAC homes.
#
#    MODIFIED   (MM/DD/YY)
#    xyuan       08/31/12 - Fix bug 14535011 - Add '-init' option
#    xyuan       08/12/12 - Fix bug 14464512
#    xyuan       04/24/12 - Add options for joining an upgraded cluster
#    sidshank    04/20/12 - adding -auto option to be used internally by root
#                           automation alone.
#    xyuan       02/20/12 - Add -prepatch & -postpatch
#    xyuan       02/02/12 - Force upgrade support
#    xyuan       01/08/12 - -oldcrshome and -version are not needed any more
#                           for downgrade command
#    xyuan       10/20/11 - XbranchMerge ksviswan_bug-12630162 from
#                           st_has_11.2.0.3.0
#    sidshank    08/11/11 - removing check_Superuser from here for bug 12739826
#    xyuan       07/27/11 - XbranchMerge xyuan_bug-12585291 from
#                           st_has_11.2.0.3.0
#    xyuan       07/27/11 - XbranchMerge xyuan_bug-12727247 from
#                           st_has_11.2.0.3.0
#    dpham       03/28/11 - New for 12c
# 

################ Documentation ################

# The SYNOPSIS section is printed out as usage when incorrect parameters
# are passed

=head1 NAME

  rootcrs.pl - Configure Oracle Clusterware

=head1 SYNOPSIS

  rootcrs.pl [-verbose] [-upgrade | -patch]
             [-paramfile <parameter-file>] 
             [-deconfig [-deinstall] [-keepdg] | -downgrade] [-force] [-lastnode]
             [-unlock [-crshome <path to CRS home>] [-nocrsstop]]
             [[-prepatch | -postpatch] [-norestart] [-nonrolling]] [-dstcrshome <path to CRS home>]
             [-join -existingnode <node name>]
             [-init]

  Options:
   -verbose    Run this script in verbose mode
   -upgrade    Oracle HA is being upgraded from previous version
   -patch      Oracle HA is being upgraded to a patch version
   -paramfile  Complete path of file specifying HA parameter values
   -lastnode   Force the node this is executing on to be considered the
               last node of the install and perform actions associated
               with configuring the last node
   -downgrade  Downgrade the clusterware
   -deconfig   Remove Oracle Clusterware to allow it to be uninstalled or reinstalled
   -deinstall  Reset the permissions on CRS home during de-configuration
   -keepdg     Keep existing diskgroups during de-configuration
   -force      Force the execution of steps in delete that cannot be verified 
	       to be safe
   -unlock     Unlock CRS home 
   -crshome    Complete path of crs home. Use with unlock option
   -nocrsstop  Used with unlock option to reset permissions on an inactive grid home
   -prepatch   Perform required steps before the Oracle patching tool (Opatch) is invoked
   -postpatch  Perform required steps after the Oracle patching tool (Opatch) is invoked
   -nonrolling Perform steps for -prepatch and -postpatch in a non-rolling fashion
   -norestart  Leave CRS stack down after patching
   -dstcrshome Complete path of CRS home for out-of-place patching only 
   -join       Add the local node to an upgraded cluster
   -existingnode An already upgraded cluster node. Must use with join option
   -init       Reset the permissions of all files and directories under CRS home


  If neither -upgrade nor -patch is supplied, a new install is performed

  To see the full manpage for this program, execute:
    perldoc rootcrs.pl

=head1 DESCRIPTION

  This script performs the operations necessary to install the Oracle
  Clusterware stack on a node of a cluster.  It must be run once on
  each node and it must be run by an authorized user, e.g. root on UNIX
  platforms, Administrator on Windows.

  Upon successful completion on each node, the Oracle Clusterware will
  be executing on that node.

=cut

################ End Documentation ################

use strict;
use English;
use File::Basename;
use File::Spec::Functions;
use Getopt::Long qw(:config no_auto_abbrev);
use Pod::Usage;

BEGIN {
   # Add the directory of this file to the search path
   push @INC, dirname($PROGRAM_NAME);
}

# root scripts modules
use crsinstall;
use crsupgrade;
use crsdeconfig;
use crspatch;
use crsutils;

my ($DEBUG, $DECONFIG, $FORCE, $LASTNODE, $UPGRADE, $DOWNGRADE, $INIT,
    $PATCH, $UNLOCK, $HELP, $DEINSTALL, $KEEPDG, $NOCRSSTOP, $AUTO) = 0;
my ($UNLOCK_HOME, $REMOTENODE);

# For patching
my ($PREPATCH, $POSTPATCH, $NONROLLING, $NORESTART) = 0;
my $DESTCRSHOME;

# For adding the local node to an upgraded cluster
my $JOIN = 0;
my $EXISTINGNODE;

# pull all parameters defined in crsconfig_params (if it exists)
# as variables in Perl
my $PARAM_FILE = catfile(dirname($0), 'crsconfig_params');

# pull all parameters defined in crsconfig_addparams
my $addfile = catfile(dirname($0), 'crsconfig_addparams');

# pull all definitions in s_crsconfig_defs (if it exists) as variables in Perl
# this file might not exist for all platforms
my $osdfile = catfile(dirname($0), 's_crsconfig_defs');

# Parse command line args
# If an incorrect option is specified, the perl POD at the beginning
# of the file is parsed into a man page
# the return code to give when the incorrect parameters are passed
my $usage_rc = 1;

GetOptions('verbose'         => \$DEBUG,
           'paramfile=s'     => \$PARAM_FILE,
           'deconfig'        => \$DECONFIG,
           'deinstall'       => \$DEINSTALL,
           'keepdg'          => \$KEEPDG,
           'patch'           => \$PATCH,
           'upgrade'         => \$UPGRADE,
           'force'           => \$FORCE,
           'lastnode'        => \$LASTNODE,
           'downgrade'       => \$DOWNGRADE,
           'unlock'          => \$UNLOCK,
           'crshome=s'       => \$UNLOCK_HOME,
           'remotenode'      => \$REMOTENODE,
           'nocrsstop!'      => \$NOCRSSTOP,
           'prepatch'        => \$PREPATCH,
           'postpatch'       => \$POSTPATCH,
           'dstcrshome=s'    => \$DESTCRSHOME,
           'nonrolling!'     => \$NONROLLING,
           'norestart!'      => \$NORESTART,
           'auto'            => \$AUTO,
           'join'            => \$JOIN,
           'existingnode=s'  => \$EXISTINGNODE,
           'init'            => \$INIT,
           'help!'           => \$HELP
          ) or pod2usage($usage_rc);

# Check validity of command line args
pod2usage(-msg => "Invalid extra options passed: @ARGV",
          -exitval => $usage_rc) if (@ARGV);

combinedArgs({"deconfig" => $DECONFIG},
  {"deinstall" => $DEINSTALL, "keepdg" => $KEEPDG}) || pod2usage(0);

combinedArgs({"prepatch" => $PREPATCH, "postpatch" => $POSTPATCH},
  {"nonrolling" => $NONROLLING, "norestart" => $NORESTART}) || pod2usage(0);

combinedArgs({"upgrade" => $UPGRADE},
  {"join" => $JOIN, "existingnode" => $EXISTINGNODE}) || pod2usage(0);

combinedArgs({"join" => $JOIN}, {"existingnode" => $EXISTINGNODE}) || pod2usage(0);

combinedArgs({"existingnode" => $EXISTINGNODE}, {"join" => $JOIN}) || pod2usage(0);

combinedArgs({"deconfig" => $DECONFIG, "downgrade" => $DOWNGRADE},
  {"lastnode" => $LASTNODE}) || pod2usage(0);

combinedArgs({"deconfig" => $DECONFIG, "downgrade" => $DOWNGRADE, "upgrade" => $UPGRADE},
  {"force" => $FORCE}) || pod2usage(0);

#-------------------------------------------------------------------------------
# MAIN SCRIPT BODY
#-------------------------------------------------------------------------------
local $SIG{'__DIE__'} = sub { dietrap(@_); };
# Catch Control-C
local $SIG{INT}       = sub { dietrap(@_); };
# Catch termination signal (kill)
# Kill signal (kill -9) can be neither trapped nor ignored
local $SIG{TERM} = sub { dietrap(@_); };

if (! $DEBUG) { $DEBUG = $ENV{'ORA_INSTALL_DEBUG'}; } 

if ($HELP) { 
   pod2usage(0);
}
elsif ($DECONFIG) { 
   crsdeconfig->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    CRSDeconfig         => $DECONFIG,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    FORCE               => $FORCE,
                    DEINSTALL           => $DEINSTALL,
                    KEEPDG              => $KEEPDG,
                    LASTNODE            => $LASTNODE,
                    crscfg_trace        => TRUE,
                    AUTO                => $AUTO
                   );
}
elsif ($UNLOCK) {
      crsutils->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    UNLOCK              => $UNLOCK,
                    UNLOCK_HOME         => $UNLOCK_HOME,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    NOCRSSTOP           => $NOCRSSTOP,
                    crscfg_trace        => TRUE
                   );

   if ($UNLOCK_HOME) {
       unlockPatchDestHome($UNLOCK_HOME);
   }
   else {
      my $exclfile = catfile($CFG->ORA_CRS_HOME,
                             'crs', 'install', 'install.excl');
      unlockCRSHomefordeinstall( $exclfile);
   }
}
elsif ($PATCH) {
      crspatch->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    CRSPatch            => $PATCH,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    DESTCRSHOME         => $DESTCRSHOME,
                    crscfg_trace        => TRUE
                   );
}
elsif ($UPGRADE) { 
    crsupgrade->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    UPGRADE             => $UPGRADE,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    FORCE               => $FORCE,
                    REMOTENODE          => $REMOTENODE,
                    JOIN                => $JOIN,
                    EXISTINGNODE        => $EXISTINGNODE,
                    crscfg_trace        => TRUE,
                    AUTO                => $AUTO
                   );
}
elsif ($DOWNGRADE) {
   crsdeconfig->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    DOWNGRADE           => $DOWNGRADE,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    FORCE               => $FORCE,
                    LASTNODE            => $LASTNODE,
                    crscfg_trace        => TRUE
                   );
}
elsif ($PREPATCH) {
      crspatch->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    PREPATCH            => $PREPATCH,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    NONROLLING          => $NONROLLING,
                    DESTCRSHOME         => $DESTCRSHOME,
                    NORESTART           => $NORESTART,
                    crscfg_trace        => TRUE
                   );
}
elsif ($POSTPATCH) {
      crspatch->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    POSTPATCH           => $POSTPATCH,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    NONROLLING          => $NONROLLING,
                    DESTCRSHOME         => $DESTCRSHOME,
                    NORESTART           => $NORESTART,
                    crscfg_trace        => TRUE
                   );
}
elsif ($INIT) {
      crsinstall->new(SIHA                => FALSE,
                      DEBUG               => $DEBUG,
                      paramfile           => $PARAM_FILE,
                      osdfile             => $osdfile,
                      addfile             => $addfile,
                      crscfg_trace        => TRUE,
                      init                => $INIT
                     );
}
else { # fresh install
    crsinstall->new(SIHA                => FALSE,
                    DEBUG               => $DEBUG,
                    paramfile           => $PARAM_FILE,
                    osdfile             => $osdfile,
                    addfile             => $addfile,
                    REMOTENODE          => $REMOTENODE,
                    crscfg_trace        => TRUE,
		    AUTO                => $AUTO
                   );
}

sub combinedArgs
{
  my %req_opts = %{$_[0]};
  my %opt_opts = %{$_[1]};

  for my $optElem (keys %opt_opts)
  {
    next if (! $opt_opts{$optElem});

    my @missingOpts;
    for my $reqElem (keys %req_opts)
    {
      if (! $req_opts{$reqElem})
      {
        push(@missingOpts, $reqElem);
      }
    }

    if (scalar(@missingOpts) == scalar(keys %req_opts))
    {
      print("One or more options required but missing: @missingOpts\n");
      return FALSE;
    }
  }

  return TRUE;
}



0;
